//
// (C) 2011 Mike Brent aka Tursi aka HarmlessLion.com
// This software is provided AS-IS. No warranty
// express or implied is provided.
//
// This notice defines the entire license for this code.
// All rights not explicity granted here are reserved by the
// author.
//
// You may redistribute this software provided the original
// archive is UNCHANGED and a link back to my web page,
// http://harmlesslion.com, is provided as the author's site.
// It is acceptable to link directly to a subpage at harmlesslion.com
// provided that page offers a URL for that purpose
//
// Source code, if available, is provided for educational purposes
// only. You are welcome to read it, learn from it, mock
// it, and hack it up - for your own use only.
//
// Please contact me before distributing derived works or
// ports so that we may work out terms. I don't mind people
// using my code but it's been outright stolen before. In all
// cases the code must maintain credit to the original author(s).
//
// -COMMERCIAL USE- Contact me first. I didn't make
// any money off it - why should you? ;) If you just learned
// something from this, then go ahead. If you just pinched
// a routine or two, let me know, I'll probably just ask
// for credit. If you want to derive a commercial tool
// or use large portions, we need to talk. ;)
//
// If this, itself, is a derived work from someone else's code,
// then their original copyrights and licenses are left intact
// and in full force.
//
// http://harmlesslion.com - visit the web page for contact info
//

#include <windows.h>
#include <stdio.h>
#include <io.h>
#include <atlstr.h>
#include <time.h>
#include "tiemul.h"
#include "diskclass.h"
#include "imagedisk.h"

// largest disk size
#define MAX_SECTORS 1440

//********************************************************
// ImageDisk
//********************************************************

// TODO: read-only today, till I am comfortable that I have it right!

// constructor
ImageDisk::ImageDisk() {
	bUseV9T9DSSD = false;
}

ImageDisk::~ImageDisk() {
}

// powerup routine
void ImageDisk::Startup() {
	BaseDisk::Startup();
}

// handle options
void ImageDisk::SetOption(int nOption, int nValue) {
	switch (nOption) {
		case OPT_IMAGE_USEV9T9DSSD:	
			bUseV9T9DSSD = nValue?true:false;
			break;

		default:
			BaseDisk::SetOption(nOption, nValue);
			break;
	}
}

bool ImageDisk::GetOption(int nOption, int &nValue) {
	switch (nOption) {
		case OPT_IMAGE_USEV9T9DSSD:	
			nValue = bUseV9T9DSSD;
			break;

		default:
			return BaseDisk::GetOption(nOption, nValue);
	}

	return true;
}

// return a local path name on the PC disk
CString ImageDisk::BuildFilename(FileInfo *pFile) {
	if (NULL == pFile) {
		return "";
	}

	// return just the disk image path
	return pDriveType[pFile->nDrive]->GetPath();
}

// Read a sector from an open file - true on success, false
// if an error occurs. buf must be at least 256 bytes!
bool ImageDisk::GetSectorFromDisk(FILE *fp, int nSector, unsigned char *buf) {
	if (NULL == fp) return false;
	if (NULL == buf) return false;

	// PC99 detection routines adapted from code by Paolo Bagnaresi! Thanks for your research, Paolo!
	// Need to add flags to disable PC99 detection (support weird disks)
	if (fseek(fp, 0, SEEK_SET)) {
		return false;
	}
	if (256 != fread(buf, 1, 256, fp)) {		// read the first block of the file
		return false;
	}
	bool bIsPC99 = (memcmp("DSK", &buf[13], 3) != 0);	// if we find DSK, then it's not PC99 for sure, if not, it probably is (or a weird disk format)
	if (bIsPC99) {
		// data to find the sector - kind of bad that we do this every time we access
		// a sector. TODO?
		int Gap1, PreIDGap, PreDatGap, SLength, SekTrack, TrkLen;

		// find the beginning of the sector by looking for 0xfe
		int idx=0;
		bool bDoubleDensity;

		while ((idx<256)&&(buf[idx]!=0xfe)) idx++;
		if (buf[idx] != 0xfe) {
			debug_write("Can't find PC99 start of sector indicator.");
			return false;
		}
		if (buf[idx+4] != 0x01) {
			debug_write("Can't find PC99 start of sector indicator!");
			return false;
		}
		if (memcmp("\xa1\xa1\xa1", &buf[idx-3], 3) == 0) {
			bDoubleDensity = true;
			if (idx != 53) {
				debug_write("Unknown track data size on PC99 disk.");
				return false;
			}
			Gap1 = 40;
			PreIDGap = 14;
			PreDatGap = 58;
			SLength = 340;
			SekTrack = 18;
			TrkLen = 6872;
		} else {
			if (memcmp("\0\0\0", &buf[idx-3], 3) != 0) {
				debug_write("Can't identify PC99 density marker");
				return false;
			}
			bDoubleDensity = false;
			if (idx != 22) {
				debug_write("Unknown track data size on PC99 disk");
				return false;
			}
			Gap1 = 16;
			PreIDGap = 7;
			PreDatGap = 31;
			SLength = 334;
			SekTrack = 9;
			TrkLen = 3253;
		}

		// now find the sector in the file - convert to side/track/sector
		int Trk = nSector / SekTrack;		// which track is it on?
		int DskSide = 0;
		if (Trk > 39) {
			Trk = 79-Trk;
			DskSide = 1;
		}
		nSector %= SekTrack;
		int nOffset = (TrkLen*40)*DskSide + Trk*TrkLen;		// assumes 40 tracks
		
		// read in the entire track, since we don't know the sector order - caching would be useful here
		// we probably should just cache the whole disk image in both cases?? If we cache the disk image,
		// then we can just read sectors in both cases. But writing becomes a tricky operation, especially
		// if we do add Omniflop support someday. Probably should just live with it?
		unsigned char *pTrk = (unsigned char*)malloc(TrkLen);
		if (fseek(fp, nOffset, SEEK_SET)) {
			debug_write("Seek failed for PC99 disk");
			free(pTrk);
			return false;
		}
		if (TrkLen != fread(pTrk, 1, TrkLen, fp)) {
			debug_write("Read failed for PC99 disk");
			free(pTrk);
			return false;
		}
		for (int tst=0; tst<SekTrack; tst++) {
			int p = Gap1 + PreIDGap + (SLength * tst);
			if ((pTrk[p] == Trk) && (pTrk[p+1] == DskSide) && (pTrk[p+2] == nSector) && (pTrk[p+3] == 1)) {
				memcpy(buf, &pTrk[Gap1 + PreDatGap + (SLength * tst)], 256);
				free(pTrk);
				return true;
			}
		}
		debug_write("PC99 disk can't find side %d, track %d, sector %d", DskSide, Trk, nSector);
		return false;
	} else {
		// V9T9 disk
		// Note: V9T9 reversed the order of side 2 of DSSD disks, so if the sector
		// range is from 360-719, we have to reverse it to get the right offset
		// (only if the option is set!) OR IS THIS A RUMOR?? I've never seen one!
		if ((bUseV9T9DSSD) && (nSector >= 360) && (nSector <= 719)) {
			debug_write("Note: using swapped V9T9 order for sector %d", nSector);
			nSector = (719-nSector) + 360;
		}

		if (fseek(fp, nSector*256, SEEK_SET)) {
			return false;
		}

		if (256 != fread(buf, 1, 256, fp)) {
			return false;
		}

		return true;
	}
}

// Write a sector to an open file - true on success, false
// Note that the file must be open for read AND write!
// if an error occurs. buf must be at least 256 bytes!
// The image on disk is updated immediately and irrevocably!
bool ImageDisk::PutSectorToDisk(FILE *fp, int nSector, unsigned char *wrbuf) {
	unsigned char buf[256];		// work buffer
	if (NULL == fp) return false;
	if (NULL == buf) return false;

	// PC99 detection routines adapted from code by Paolo Bagnaresi! Thanks for your research, Paolo!
	// Need to add flags to disable PC99 detection (support weird disks)
	if (fseek(fp, 0, SEEK_SET)) {
		return false;
	}
	if (256 != fread(buf, 1, 256, fp)) {		// read the first block of the file
		return false;
	}
	bool bIsPC99 = (memcmp("DSK", &buf[13], 3) != 0);	// if we find DSK, then it's not PC99 for sure, if not, it probably is (or a weird disk format)
	if (bIsPC99) {
		// data to find the sector - kind of bad that we do this every time we access
		// a sector. TODO?
		int Gap1, PreIDGap, PreDatGap, SLength, SekTrack, TrkLen;

		// find the beginning of the sector by looking for 0xfe
		int idx=0;
		bool bDoubleDensity;

		while ((idx<256)&&(buf[idx]!=0xfe)) idx++;
		if (buf[idx] != 0xfe) {
			debug_write("WCan't find PC99 start of sector indicator.");
			return false;
		}
		if (buf[idx+4] != 0x01) {
			debug_write("Can't find PC99 start of sector indicator!");
			return false;
		}
		if (memcmp("\xa1\xa1\xa1", &buf[idx-3], 3) == 0) {
			bDoubleDensity = true;
			if (idx != 53) {
				debug_write("WUnknown track data size on PC99 disk.");
				return false;
			}
			Gap1 = 40;
			PreIDGap = 14;
			PreDatGap = 58;
			SLength = 340;
			SekTrack = 18;
			TrkLen = 6872;
		} else {
			if (memcmp("\0\0\0", &buf[idx-3], 3) != 0) {
				debug_write("WCan't identify PC99 density marker");
				return false;
			}
			bDoubleDensity = false;
			if (idx != 22) {
				debug_write("WUnknown track data size on PC99 disk");
				return false;
			}
			Gap1 = 16;
			PreIDGap = 7;
			PreDatGap = 31;
			SLength = 334;
			SekTrack = 9;
			TrkLen = 3253;
		}

		// now find the sector in the file - convert to side/track/sector
		int Trk = nSector / SekTrack;		// which track is it on?
		int DskSide = 0;
		if (Trk > 39) {
			Trk = 79-Trk;
			DskSide = 1;
		}
		nSector %= SekTrack;
		int nOffset = (TrkLen*40)*DskSide + Trk*TrkLen;		// assumes 40 tracks
		
		// read in the entire track, since we don't know the sector order - caching would be useful here
		// we probably should just cache the whole disk image in both cases?? If we cache the disk image,
		// then we can just read sectors in both cases. But writing becomes a tricky operation, especially
		// if we do add Omniflop support someday. Probably should just live with it?
		// Note that by cache, I mean like files are cached, by just reading the sectors into a buffer
		// so that we know where each one is exactly. But this is probably a bad idea -- it makes my life
		// easier, but there is high risk of the user changes the disk image of overwriting an entire disk
		// just because a file was changed. Mind you, for the sake of file updates, maybe it makes sense
		// to cache the disk while updating the bitmap just to reduce writes? Hell, I dunno. Too many
		// tradeoffs to balance.
		unsigned char *pTrk = (unsigned char*)malloc(TrkLen);
		if (fseek(fp, nOffset, SEEK_SET)) {
			debug_write("WSeek failed for PC99 disk");
			free(pTrk);
			return false;
		}
		if (TrkLen != fread(pTrk, 1, TrkLen, fp)) {
			debug_write("WRead failed for PC99 disk");
			free(pTrk);
			return false;
		}
		for (int tst=0; tst<SekTrack; tst++) {
			int p = Gap1 + PreIDGap + (SLength * tst);
			if ((pTrk[p] == Trk) && (pTrk[p+1] == DskSide) && (pTrk[p+2] == nSector) && (pTrk[p+3] == 1)) {
				// Just write back the one sector, no need to mess with the whole track
				if (fseek(fp, nOffset + Gap1 + PreDatGap + (SLength * tst), SEEK_SET)) {
					debug_write("WSeek failed for PC99 disk");
					free(pTrk);
					return false;
				}
				if (256 != fwrite(wrbuf, 1, 256, fp)) {
					debug_write("WFailed to write entire sector - corruption may have occurred.");
					free(pTrk);
					return false;
				}

				free(pTrk);
				return true;
			}
		}
		debug_write("WPC99 disk can't find side %d, track %d, sector %d", DskSide, Trk, nSector);
		return false;
	} else {
		// V9T9 disk
		// Note: V9T9 reversed the order of side 2 of DSSD disks, so if the sector
		// range is from 360-719, we have to reverse it to get the right offset
		// (only if the option is set!) OR IS THIS A RUMOR?? I've never seen one!
		if ((bUseV9T9DSSD) && (nSector >= 360) && (nSector <= 719)) {
			debug_write("WNote: using swapped V9T9 order for sector %d", nSector);
			nSector = (719-nSector) + 360;
		}

		if (fseek(fp, nSector*256, SEEK_SET)) {
			return false;
		}

		if (256 != fwrite(wrbuf, 1, 256, fp)) {
			return false;
		}

		return true;
	}
}

// try to locate the FDR for a file on an open disk image
// return true if found, the sector index is filled into pFile->nLocalData and the data stored in fdr (must be 256 bytes!)
// false if not found, fdr buffer is overwritten
bool ImageDisk::FindFileFDR(FILE *fp, FileInfo *pFile, unsigned char *fdr) {
	// We aren't going to get fancy, we won't search the file table in order. We'll 
	// just start at the beginning and run through it all. This means the tricks on
	// Thierry's page won't work here.
	unsigned char sector1[256];	// work buffer
	if (!GetSectorFromDisk(fp, 1, sector1)) {
		fclose(fp);
		debug_write("Can't read sector 1 from %s, errno %d", (LPCSTR)BuildFilename(pFile), errno);
		pFile->LastError = ERR_DEVICEERROR;
		return false;
	}

	// loop until no more files, a disk error, or we find it
	CString csSearch = pFile->csName + "          ";
	csSearch.Truncate(10);
	for (int i=0; ; i++) {
		int nSect = sector1[i*2]*256 + sector1[i*2+1];
		if (nSect == 0) {
			// file not found
			fclose(fp);
			debug_write("Can't find file %s on %s.", pFile->csName, (LPCSTR)BuildFilename(pFile));
			pFile->LastError = ERR_FILEERROR;
			return false;
		}

		if (!GetSectorFromDisk(fp, nSect, fdr)) {
			fclose(fp);
			debug_write("Can't read sector %d (file %d) from %s, errno %d", nSect, i, (LPCSTR)BuildFilename(pFile), errno);
			pFile->LastError = ERR_DEVICEERROR;
			return false;
		}

		// got a supposed FDR, try to match the filename
		if (memcmp(fdr, (LPCSTR)csSearch, csSearch.GetLength()) == 0) {
			// it's a match!
			pFile->nLocalData = nSect;		// save off the FDR location
			break;
		}
	}

	return true;
}

// copy file information from an FDR buffer (must be 256 bytes)
// into the FileInfo structure
void ImageDisk::CopyFDRToFileInfo(unsigned char *buf, FileInfo *pFile) {
	pFile->ImageType = IMAGE_SECTORDSK;
	// fill in the information 
	pFile->LengthSectors=(buf[0x0e]<<8)|buf[0x0f];
	pFile->FileType=buf[0x0c];
	pFile->RecordsPerSector=buf[0x0d];
	pFile->BytesInLastSector=buf[0x10];
	pFile->RecordLength=buf[0x11];
	pFile->NumberRecords=(buf[0x13]<<8)|buf[0x12];		// NOTE: swapped on disk!
	// translate FileType to Status
	pFile->Status = 0;
	if (pFile->FileType & TIFILES_VARIABLE) pFile->Status|=FLAG_VARIABLE;
	if (pFile->FileType & TIFILES_INTERNAL) pFile->Status|=FLAG_INTERNAL;
}

// Open an existing file, check the header against the parameters
bool ImageDisk::TryOpenFile(FileInfo *pFile) {
	char *pMode;
	unsigned char fdr[256];	// work buffer

	CString csFileName = BuildFilename(pFile);
	int nMode = pFile->Status & FLAG_MODEMASK;	// should be UPDATE, APPEND or INPUT
	FILE *fp=NULL;
	FileInfo lclInfo;

	switch (nMode) {
		case FLAG_UPDATE:
			pMode="update";
			fp=fopen(csFileName, "r+b");
			break;

#if 0
	// TODO: for now, we only support reading
	// But we kept UPDATE because some programs are sloppy and
	// use that mode because it's the default.
		case FLAG_APPEND:
			pMode="append";
			fp=fopen(csFileName, "ab");
			break;
#else
		case FLAG_APPEND:
			pMode="append";
			debug_write("Writing to disk images not yet supported.");
			break;
#endif

		case FLAG_INPUT:
			pMode="input";
			fp=fopen(csFileName, "rb");
			break;

		default:
			debug_write("Unknown mode - can't open.");
			pMode="unknown";
	}

	if (NULL == fp) {
		debug_write("Can't open %s for %s, errno %d.", (LPCSTR)csFileName, pMode, errno);
		pFile->LastError = ERR_FILEERROR;
		return false;
	}

	// we got the disk, now we need to find the file.
	if (!FindFileFDR(fp, pFile, fdr)) {
		return false;
	}
	fclose(fp);

	// now we have a file, and an FDR, so fill in the data
	// there should be no data to copy as we are just opening the file
	lclInfo.CopyFileInfo(pFile, false);
	CopyFDRToFileInfo(fdr, &lclInfo);

	if (lclInfo.ImageType == IMAGE_UNKNOWN) {
		debug_write("%s is an unknown file type - can not open.", (LPCSTR)lclInfo.csName);
		pFile->LastError = ERR_BADATTRIBUTE;
		return false;
	}

	// only verify on open (we reuse this function for sector reads)
	if ((pFile->OpCode == OP_OPEN) || (pFile->OpCode == OP_LOAD)) {
		// Verify the parameters as a last step before we OK it all :)
		if ((pFile->FileType&TIFILES_MASK) != (lclInfo.FileType&TIFILES_MASK)) {
			debug_write("Incorrect file type: %d/%s (real) vs %d/%s (requested)", lclInfo.FileType&TIFILES_MASK, GetAttributes(lclInfo.FileType), pFile->FileType&TIFILES_MASK, GetAttributes(pFile->FileType));
			pFile->LastError = ERR_BADATTRIBUTE;
			return false;
		}

		if (0 == (lclInfo.FileType & TIFILES_PROGRAM)) {
			// check record length (we already verified if PROGRAM was wanted above)

			if (pFile->RecordLength == 0) {
				pFile->RecordLength = lclInfo.RecordLength;
			}

			if (pFile->RecordLength != lclInfo.RecordLength) {
				debug_write("Record Length mismatch: %d (real) vs %d (requested)", lclInfo.RecordLength, pFile->RecordLength);
				pFile->LastError = ERR_BADATTRIBUTE;
				return false;
			}
		}
	}

	// seems okay? Copy the data over from the PAB
	pFile->CopyFileInfo(&lclInfo, false);
	return true;
}

// Read the file into the disk buffer
// This function's job is to read the file into individual records
// into the memory buffer so it can be worked on generically. This
// function is not used for PROGRAM image files, but everything else
// is fair game. It reads the entire file, and it stores the records
// at maximum size (even for variable length files). Since TI files
// should max out at about 360k (largest floppy size), this should
// be fine (even though hard drives do exist, there are very few
// files for them, and thanks to MESS changing all the time the
// format seems not to be well defined.) Note that if you DO open
// a large file, though, Classic99 will try to allocate enough RAM
// to hold it. Buyer beware, so to speak. ;)
bool ImageDisk::BufferFile(FileInfo *pFile) {
	// this checks for PROGRAM images as well as Classic99 sequence bugs
	if (0 == pFile->RecordLength) {
		debug_write("Attempting to buffer file with 0 record length, can't do it.", (LPCSTR)pFile->csName);
		return false;
	}

	// all right, then, let's give it a shot.
	// So really, all we do here is branch out to the correct function
	switch (pFile->ImageType) {
		case IMAGE_UNKNOWN:
			debug_write("Attempting to buffer unknown file type %s, can't do it.", (LPCSTR)pFile->csName);
			break;

		case IMAGE_SECTORDSK:
			return BufferSectorFile(pFile);

		default:
			debug_write("Failed to buffer undetermined file type %d for %s", pFile->ImageType, (LPCSTR)pFile->csName);
	}
	return false;
}

// parse the cluster list in an FDR (256 byte sector)
// Returns a malloc'd list of ints, ending with 0 (user frees)
int *ImageDisk::ParseClusterList(unsigned char *fdr) {
	int *pList = (int*)malloc(sizeof(int)*MAX_SECTORS+1);
	int nListPos = 0;
	int nOff = 0x1c;
	int nFilePos = 0;

	while (nOff < 0x100) {
		int um, sn, of, num, ofs;

		um = fdr[nOff]&0xff;
		sn = fdr[nOff+1]&0xff;
		of = fdr[nOff+2]&0xff;
		nOff+=3;

		num = ((sn&0x0f)<<8) | um;
		ofs = (of<<4) | ((sn&0xf0)>>4);

		if (num == 0) {
			break;
		}

		for (int i=nFilePos; i<=ofs; i++) {
			pList[nListPos++] = num++;
			if (nListPos >= MAX_SECTORS) {
				debug_write("Cluster list exceeds maximum number of sectors, aborting.");
				free(pList);
				return NULL;
			}
		}

		nFilePos = ofs;
	}

	pList[nListPos] = 0;

	return pList;
}

// Buffer a sector disk style file - pFile->nLocalData
// contains the index of the FDR record. This will
// read somewhat similar to how Fiad files read, actually.
bool ImageDisk::BufferSectorFile(FileInfo *pFile) {
	int idx, nSector;
	unsigned char *pData;
	unsigned char tmpbuf[256];
	int *pSectorList = NULL;
	int nSectorPos = 0;

	// Fixed records are obvious. Variable length records are prefixed
	// with a byte that indicates how many bytes are in this record. It's
	// all padded to 256 byte blocks. If it won't fit, the space in the
	// rest of the sector is wasted (and 0xff marks it)
	// Even better - the NumberRecords field in a variable file is a lie,
	// it's really a sector count. So we need to read them differently,
	// more like a text file.
	CString csFileName=BuildFilename(pFile);
	FILE *fp=fopen(csFileName, "rb");
	if (NULL == fp) {
		debug_write("Failed to open %s", (LPCSTR)csFileName);
		return false;
	}

	// read in the FDR so we can get the cluster list
	if (!GetSectorFromDisk(fp, pFile->nLocalData, tmpbuf)) {
		fclose(fp);
		debug_write("Failed to retrieve FDR at sector %d for %s", pFile->nLocalData, csFileName);
		return false;
	}
	// Get the sector list by parsing the Cluster list (0 terminated)
	// Then we don't need the FDR anymore.
	pSectorList = ParseClusterList(tmpbuf);
	if (NULL == pSectorList) {
		fclose(fp);
		debug_write("Failed to parse cluster list for %s", csFileName);
		return false;
	}

	// get a new buffer as well sized as we can figure
	if (NULL != pFile->pData) {
		free(pFile->pData);
		pFile->pData=NULL;
	}

	// Datasize = (number of records+10) * (record size + 2)
	// the +10 gives it a little room to grow
	// the +2 gives room for a length word (16bit) at the beginning of each
	// record, necessary because it may contain binary data with zeros
	if (pFile->Status & FLAG_VARIABLE) {
		// like with the text files, we'll just assume a generic buffer size of 
		// about 100 records and grow it if we need to :)
		pFile->nDataSize = (100) * (pFile->RecordLength + 2);
		pFile->pData = (unsigned char*)malloc(pFile->nDataSize);
	} else {
		// for fixed length fields we know how much memory we need
		pFile->nDataSize = (pFile->NumberRecords+10) * (pFile->RecordLength + 2);
		pFile->pData = (unsigned char*)malloc(pFile->nDataSize);
	}

	idx=0;							// count up the records read
	nSector = 0;					// position in this sector
	nSectorPos = 0;					// position in the sector list
	pData = pFile->pData;

	if (pSectorList[0] == 0) {
		// empty file - we're done
		fclose(fp);
		free(pSectorList);
		debug_write("%s::%s was empty.", (LPCSTR)csFileName, (LPCSTR)pFile->csName);
		pFile->NumberRecords = 0;
		return true;
	}

	if (!GetSectorFromDisk(fp, pSectorList[nSectorPos++], tmpbuf)) {
		fclose(fp);
		debug_write("Failed reading first sector %d in file %s", pSectorList[nSectorPos-1], pFile->csName);
		free(pSectorList);
		return false;
	}

	// we need to let the embedded code decide the terminating rule
	for (;;) {
		if (pFile->Status & FLAG_VARIABLE) {
			// read a variable record
			int nLen=tmpbuf[nSector++];
			if (nLen==0xff) {
				// end of sector indicator, no record read, skip rest of sector
				nSector=0;
				pFile->NumberRecords--;
				// are we done?
				if (pFile->NumberRecords == 0) {
					// yes we are, get the true count
					pFile->NumberRecords = idx;
					break;
				}
				// otherwise, read in the next sector in the list
				if (pSectorList[nSectorPos] == 0) {
					debug_write("Read past EOF - truncating read.");
					pFile->NumberRecords = idx;
					break;
				}
				if (!GetSectorFromDisk(fp, pSectorList[nSectorPos++], tmpbuf)) {
					debug_write("Failed reading sector %d in file %s - truncating", pSectorList[nSectorPos-1], pFile->csName);
					pFile->NumberRecords = idx;
					break;
				}
			} else {
				// check for buffer resize
				if ((pFile->pData+pFile->nDataSize) - pData < (pFile->RecordLength+2)*10) {
					int nOffset = pData - pFile->pData;		// in case the buffer moves
					// time to grow the buffer - add another 100 lines
					pFile->nDataSize += (100) * (pFile->RecordLength + 2);
					pFile->pData  = (unsigned char*)realloc(pFile->pData, pFile->nDataSize);
					pData = pFile->pData + nOffset;
				}
				
				// clear buffer
				memset(pData, 0, pFile->RecordLength+2);

				// check again
				if (256-nSector < nLen) {
					debug_write("Corrupted file - truncating read.");
					pFile->NumberRecords = idx;
					break;
				}

				// we got some data, read it in and count off the record
				// verify it (don't get screwed up by a bad file)
				if (nLen > pFile->RecordLength) {
					debug_write("Potentially corrupt file - skipping end of record.");
					
					// store length data
					*(unsigned short*)pData = pFile->RecordLength;
					pData+=2;

					memcpy(pData, &tmpbuf[nSector], pFile->RecordLength);
					nSector+=nLen;
					// trim down nLen
					nLen = pFile->RecordLength;
				} else {
					// record is okay (normal case)
					
					// write length data
					*(unsigned short*)pData = nLen;
					pData+=2;

					memcpy(pData, &tmpbuf[nSector], nLen);
					nSector+=nLen;
				}
				// count off a valid record and update the pointer
				idx++;
				pData+=pFile->RecordLength;
			}
		} else {
			// are we done?
			if (idx >= pFile->NumberRecords) {
				break;
			}

			// clear buffer
			memset(pData, 0, pFile->RecordLength+2);

			// read a fixed record
			if (256-nSector < pFile->RecordLength) {
				// not enough room for another record, skip to the next sector
				if (pSectorList[nSectorPos] == 0) {
					debug_write("Read past EOF - truncating read.");
					pFile->NumberRecords = idx;
					break;
				}
				if (!GetSectorFromDisk(fp, pSectorList[nSectorPos++], tmpbuf)) {
					debug_write("Failed reading sector %d in file %s - truncating", pSectorList[nSectorPos-1], pFile->csName);
					pFile->NumberRecords = idx;
					break;
				}
				nSector=0;
			} else {
				// a little simpler, we just need to read the data
				*(unsigned short*)pData = pFile->RecordLength;
				pData+=2;

				memcpy(pData, &tmpbuf[nSector], pFile->RecordLength);
				nSector += pFile->RecordLength;
				idx++;
				pData += pFile->RecordLength;
			}
		}
	}

	fclose(fp);
	free(pSectorList);
	debug_write("%s::%s read %d records", (LPCSTR)csFileName, (LPCSTR)pFile->csName, pFile->NumberRecords);
	return true;
}

// retrieve the disk name from a disk image - relatively
// straight forward, it's the first 10 characters of the disk in sector 0
CString ImageDisk::GetDiskName() {
	unsigned char buf[256];
	CString csDiskName;

	CString csFileName=GetPath();
	FILE *fp=fopen(csFileName, "rb");
	if (NULL == fp) {
		debug_write("Failed to open %s", (LPCSTR)csFileName);
		return BAD_DISK_NAME;
	}

	// read in sector 0
	if (!GetSectorFromDisk(fp, 0, buf)) {
		fclose(fp);
		debug_write("Failed to retrieve sector 0 for %s", csFileName);
		return BAD_DISK_NAME;
	}

	// and parse out the diskname
	for (int idx=0; idx<10; idx++) {
		if (buf[idx] == ' ') break;
		csDiskName+=buf[idx];
	}

	return csDiskName;
}

// Open a file with a particular mode, creating it if necessary
// TODO: no write modes are supported today
FileInfo *ImageDisk::Open(FileInfo *pFile) {
	FileInfo *pNewFile=NULL;
	CString csTmp;

	if (pFile->bOpen) {
		// trying to open a file that is already open! Can't allow that!
		pFile->LastError = ERR_FILEERROR;
		return NULL;
	}

	// See if we can get a new file handle from the driver
	pNewFile = pDriveType[pFile->nDrive]->AllocateFileInfo();
	if (NULL == pNewFile) {
		// no files free
		pFile->LastError = ERR_BUFFERFULL;
		return NULL;
	}

	if (pFile->Status & FLAG_VARIABLE) {
		// variable length file - check maximum length
		if (pFile->RecordLength > 254) {
			pFile->LastError = ERR_BADATTRIBUTE;
			goto error;
		}
	}

	// So far so good -- let's try to actually find the file
	// We need to find the file, verify it is correct to open it,
	// then load all its data into a buffer.

	// Check for directory (empty filename)
	if (pFile->csName.GetLength() == 0) {
		char *pData, *pStart;

		// read directory!
		if ((pFile->Status & FLAG_MODEMASK) != FLAG_INPUT) {
			debug_write("Can't open directory for anything but input");
			pFile->LastError = ERR_ILLEGALOPERATION;
			goto error;
		}
		// check for 0, map to actual size (38)
		if (pFile->RecordLength == 0) {
			pFile->RecordLength = 38;
		}
		// internal fixed 38
		if (((pFile->Status & FLAG_TYPEMASK) != FLAG_INTERNAL) || (pFile->RecordLength != 38)) {
			debug_write("Must open directory as IF38");
			pFile->LastError = ERR_BADATTRIBUTE;
			goto error;
		}
		// Well, it's good then, buffer and format the directory records
		FileInfo *Filenames = NULL;
		pFile->NumberRecords = GetDirectory(pFile, Filenames);

		// get a new buffer as well sized as we can figure
		if (NULL != pFile->pData) {
			free(pFile->pData);
			pFile->pData=NULL;
		}

		// sadly we can't make /this/ directory support longer names either,
		// because most applications that use it either hardcode the 38 or
		// they assume the filename is always 10 characters long. We can
		// return the short filename though! :)

		// for fixed length fields we know how much memory we need
		// we add one record for the terminator
		pFile->nDataSize = (pFile->NumberRecords+1) * (pFile->RecordLength + 2);
		pFile->pData = (unsigned char*)malloc(pFile->nDataSize);
		pData = (char*)pFile->pData;

		// Record 0 contains:
		// - Diskname (an ascii string of upto 10 chars).
		// - The number zero.
		// - The number of sectors on disk (as 
		// - The number of free sectors on disk.
		memset(pData, 0, pFile->RecordLength+2);

		// DiskName - first 10 bytes - we provide the local path (last 10 chars of it)
		pStart=pData;
		*(unsigned short*)pData = (unsigned short)38;
		pData+=2;
		*(pData++) = Filenames[0].csName.GetLength();
		memcpy(pData, (LPCSTR)Filenames[0].csName, Filenames[0].csName.GetLength());
		pData+=Filenames[0].csName.GetLength();
		pData=WriteAsFloat(pData,0);	// always 0
		pData=WriteAsFloat(pData,Filenames[0].LengthSectors);		// number of sectors on the disk
		pData=WriteAsFloat(pData,Filenames[0].BytesInLastSector);	// number of free sectors on the disk

		// now the rest of the entries. 
		// - Filename (an ascii string of upto 10 chars) 
		// - Filetype: 1=D/F, 2=D/V, 3=I/F, 4=I/V, 5=Prog, 0=end of directory.
		//   If the file is protected, this number is negative (-1=D/F, etc).
		// - File size in sectors (including the FDR itself).
		// - File record length (0 for programs).
		for (int idx=1; idx<pFile->NumberRecords; idx++) {
			pData = pStart + pFile->RecordLength+2;
			pStart = pData;

			memset(pData, 0, pFile->RecordLength+2);
			csTmp = Filenames[idx].csName;
			csTmp = csTmp.Left(10);
			*(unsigned short*)pData = (unsigned short)38;
			pData+=2;
			*(pData++) = csTmp.GetLength();
			memcpy(pData, (LPCSTR)csTmp, csTmp.GetLength());
			pData+=csTmp.GetLength();
			int nType=0;
			if (Filenames[idx].FileType & TIFILES_PROGRAM) {
				nType = 5;
			} else {
				nType = 1;	// DF
				if (Filenames[idx].FileType & TIFILES_INTERNAL) {
					nType+=2;
				}
				if (Filenames[idx].FileType & TIFILES_VARIABLE) {
					nType+=1;
				}
			}
			pData=WriteAsFloat(pData, nType);
			pData=WriteAsFloat(pData, Filenames[idx].LengthSectors);
			pData=WriteAsFloat(pData, Filenames[idx].RecordLength);
		}

		// free the array
		free(Filenames);
		Filenames=NULL;

		// and last, the terminator
		pData = pStart + pFile->RecordLength+2;
		pStart = pData;

		memset(pData, 0, pFile->RecordLength+2);
		csTmp = "";
		*(unsigned short*)pData = (unsigned short)38;
		pData+=2;
		*(pData++) = csTmp.GetLength();
		memcpy(pData, (LPCSTR)csTmp, csTmp.GetLength());
		pData+=csTmp.GetLength();
		pData=WriteAsFloat(pData, 0);
		pData=WriteAsFloat(pData, 0);
		pData=WriteAsFloat(pData, 0);

		// now add two to the number of records (header and terminator)
		pFile->NumberRecords+=2;

		// that should do it!
	} else {
		// let's see what we are doing here...
		switch (pFile->Status & FLAG_MODEMASK) {
			case FLAG_OUTPUT:
				if (!CreateOutputFile(pFile)) {
					goto error;
				}
				break;

			// we can open these both the same way
			case FLAG_UPDATE:
			case FLAG_APPEND:
				// First, try to open the file
				if (!TryOpenFile(pFile)) {
					// No? Try to create it then?
					if (pFile->LastError == ERR_FILEERROR) {
						if (!CreateOutputFile(pFile)) {
							// Still no? Error out then (save error already set)
							goto error;
						}
					} else {
						goto error;
					}
				}

				// So we should have a file now - read it in
				if (!BufferFile(pFile)) {
					pFile->LastError = ERR_FILEERROR;
					goto error;
				}
				break;

			default:	// should only be FLAG_INPUT
				if (!TryOpenFile(pFile)) {
					// Error out then, must exist for input (save old error)
					goto error;
				}

				// So we should have a file now - read it in
				if (!BufferFile(pFile)) {
					pFile->LastError = ERR_FILEERROR;
					goto error;
				}
				break;
		}
	}

	// Finally, transfer the object over to the DriveType object
	pNewFile->CopyFileInfo(pFile, false);
	return pNewFile;

error:
	// release the allocated fileinfo, we didn't succeed to open
	// Use the base class as we have nothing to flush
	Close(pNewFile);
	return NULL;
}

// Load a PROGRAM image file - this happens immediately and doesn't
// need to use a buffer (nor does it close)
bool ImageDisk::Load(FileInfo *pFile) {
	FILE *fp;
	int read_bytes;
	CString csFileName = BuildFilename(pFile);

	// sanity check -- make sure we don't request more data
	// than there is RAM. A real TI would probably wrap the 
	// address counter, but for simplicity we don't. It's
	// likely a bug anyway! If we want to do emulator proof
	// code, though... ;)
	if (pFile->DataBuffer + pFile->RecordNumber > 0x4000) {
		debug_write("Attempt to load bytes past end of VDP, truncating");
		pFile->RecordNumber = 0x4000 - pFile->DataBuffer;
	}

	// We may need to fill in some of the FileInfo object here
	pFile->Status = FLAG_INPUT;
	pFile->FileType = TIFILES_PROGRAM;
	
	if (!TryOpenFile(pFile)) {
		// couldn't open the file - keep original error code
		return false;
	}

	// XB first tries to load as a PROGRAM image file with the
	// maximum available VDP RAM. If that fails with code 0x60,
	// it tries again as the proper DIS/FIX254
	// I am leaving this comment here, but hacks might not be
	// needed anymore now that the headers are properly tested.

	// It's good, so we try to open it now
	fp=fopen(csFileName, "rb");
	if (NULL == fp) {
		debug_write("Late failure opening %s", (LPCSTR)csFileName);	
		return false;
	}
	
	// we need to get the disk sector list
	unsigned char fdr[256];
	if (!GetSectorFromDisk(fp, pFile->nLocalData, fdr)) {
		fclose(fp);
		debug_write("Late failure reading FDR at sector %d on %s", pFile->nLocalData, (LPCSTR)csFileName);	
		return false;
	}
	int *pSectors = ParseClusterList(fdr);
	if (NULL == pSectors) {
		fclose(fp);
		debug_write("Late failure parsing cluster list in FDR at sector %d on %s", pFile->nLocalData, (LPCSTR)csFileName);	
		return false;
	}

	// now we run through the sectors and load the data up to the maximum requested,
	// or the end of the file, whichever comes first. The last sector only needs to
	// honor the BytesInLastSector field.
	read_bytes = 0;
	int VDPOffset = pFile->DataBuffer;
	int nBytesLeft = pFile->RecordNumber;
	for (int i=0; i < pFile->LengthSectors; i++) {
		unsigned char tmpbuf[256];
		int nToRead;

		int pos = pSectors[i];
		if (pos == 0) {
			break;
		}
		if (!GetSectorFromDisk(fp, pos, tmpbuf)) {
			debug_write("Error reading sector %d for %s - truncating.", pos, (LPCSTR)csFileName);
			break;
		}
		if (i == pFile->LengthSectors-1) {
			// last sector
			nToRead = pFile->BytesInLastSector;
			if (nToRead == 0) {
				nToRead = 256;
			}
		} else {
			nToRead = 256;
		}
		if (nToRead > nBytesLeft) {
			nToRead = nBytesLeft;
		}
		memcpy(&VDP[VDPOffset], tmpbuf, nToRead);

		// update heatmap
		for (int idx=0; idx<nToRead; idx++) {
			UpdateHeatVDP(VDPOffset+idx);
		}

		read_bytes+=nToRead;
		nBytesLeft-=nToRead;
		VDPOffset+=nToRead;

		if (nBytesLeft < 1) {
			break;
		}
	}

	free(pSectors);
	debug_write("loading 0x%X bytes", read_bytes);	// do we need to give this value to the user?
	fclose(fp);										// all done

	// handle DSK1 automapping (AutomapDSK checks whether it's enabled)
	AutomapDSK(&VDP[pFile->DataBuffer], pFile->RecordNumber, pFile->nDrive, false);

	return true;
}

// SBRLNK opcodes (files is handled by shared handler)

// Get a list of the files in the current directory (up to 127)
// returns the count of files, and an allocated array at Filenames
// The array is a list of FileInfo objects. Note that unlike the
// FIAD version, this one creates record 0 as well from the disk
// header. (Thus returns up to 128 records)
// Caller must free Filenames!
int ImageDisk::GetDirectory(FileInfo *pFile, FileInfo *&Filenames) {
	int n;
	unsigned char sector1[256];
	CString csFilename = BuildFilename(pFile);
	
	FILE *fp = fopen(csFilename, "rb");
	if (NULL == fp) {
		debug_write("Could not open disk %s", csFilename);
		return 0;
	}

	if (!GetSectorFromDisk(fp, 0, sector1)) {
		debug_write("Could not read sector 0 from %s", csFilename);
		fclose(fp);
		return 0;
	}

	Filenames = (FileInfo*)malloc(sizeof(FileInfo) * 128);
	if (NULL == Filenames) {
		debug_write("Couldn't malloc memory for directory.");
		fclose(fp);
		return 0;
	}
	for (int idx=0; idx<128; idx++) {
		new(&Filenames[idx]) FileInfo;
	}

	// Record 0 contains:
	// - Diskname (an ascii string of upto 10 chars).
	// - The number zero.
	// - The number of sectors on disk.
	// - The number of free sectors on disk.
	// A little hacky, but this will work - 0x14 is unused
	sector1[0x14]='\0';
	Filenames[0].csName = sector1;
	Filenames[0].csName.Truncate(10);
	Filenames[0].LengthSectors = sector1[0x0a]*256+sector1[0x0b];	// sectors total
	Filenames[0].BytesInLastSector = Filenames[0].LengthSectors;	// sectors free (after we subtract the bitmap below
	for (int i=0; i<(Filenames[0].LengthSectors+7)/8; i++) {
		// calculate free sectors from the sector bitmap
		unsigned char x=sector1[0x38+i];
		for (unsigned char j=0x80; j>0; j>>=1) {
			if (x&j) Filenames[0].BytesInLastSector--;
		}
	}

	if (!GetSectorFromDisk(fp, 1, sector1)) {
		debug_write("Could not read sector 1 from %s", csFilename);
		fclose(fp);
		return 1;
	}

	n=1;	// start at 1 because 0 is the diskname
	for (;;) {
		unsigned char tmpbuf[256];
		int nSect = sector1[(n-1)*2]*256 + sector1[(n-1)*2+1];		// read FDR entry
		if (nSect == 0) {
			// list finished
			break;
		}
		if (GetSectorFromDisk(fp, nSect, tmpbuf)) {
			// got an FDR - get the filename from it
			tmpbuf[0x10] = '\0';	// supposed to be anyway, this lets us treat the filename as a string
			Filenames[n].csName = tmpbuf;

			// and get the fileinfo data too
			CopyFDRToFileInfo(tmpbuf, &Filenames[n]);

			if (IMAGE_UNKNOWN != Filenames[n].ImageType) {
				n++;
				if (n>126) {
					break;
				}
			}
		}
	}

	fclose(fp);
	return n;
}

// ReadSector: nDrive=Drive#, DataBuffer=VDP address, RecordNumber=Sector to read
// Must return the sector number in RecordNumber if no error.
bool ImageDisk::ReadSector(FileInfo *pFile) {
	bool nRet = true;

	// sanity test
	if (pFile->DataBuffer + 256 > 0x4000) {
		debug_write("Attempt to read sector past end of VDP memory, aborting.");
		pFile->LastError = ERR_DEVICEERROR;
		return false;
	}

	// Zero buffer first
	memset(&VDP[pFile->DataBuffer], 0, 256);

	// any sector is okay now!
	CString csPath = BuildFilename(pFile);
	FILE *fp = fopen(csPath, "rb");
	if (NULL == fp) {
		debug_write("Can't open %s for sector read.", (LPCSTR)csPath);
		pFile->LastError = ERR_DEVICEERROR;
		return false;
	}
	if (!GetSectorFromDisk(fp, pFile->RecordNumber, &VDP[pFile->DataBuffer])) {
		debug_write("Can't read sector %d on %s.", pFile->RecordNumber, (LPCSTR)csPath);
		fclose(fp);
		pFile->LastError = ERR_DEVICEERROR;
		return false;
	}

	// it wants the entire sector, so we wrote it to the appropriate spot
	fclose(fp);
	return true;
}

// ReadSector: nDrive=Drive#, DataBuffer=VDP address, RecordNumber=Sector to read
// Must return the sector number in RecordNumber if no error.
bool ImageDisk::WriteSector(FileInfo *pFile) {
	bool nRet = true;

	// sanity test
	if (pFile->DataBuffer + 256 > 0x4000) {
		debug_write("Attempt to write sector from buffer past end of VDP memory, aborting.");
		pFile->LastError = ERR_DEVICEERROR;
		return false;
	}

	// any sector is okay now!
	CString csPath = BuildFilename(pFile);
	FILE *fp = fopen(csPath, "rb+");		// must be able to read and write to update the image
	if (NULL == fp) {
		debug_write("Can't open %s for sector write.", (LPCSTR)csPath);
		pFile->LastError = ERR_DEVICEERROR;
		return false;
	}
	if (!PutSectorToDisk(fp, pFile->RecordNumber, &VDP[pFile->DataBuffer])) {
		debug_write("Can't write sector %d on %s.", pFile->RecordNumber, (LPCSTR)csPath);
		fclose(fp);
		pFile->LastError = ERR_DEVICEERROR;
		return false;
	}

	// wrote the sector as requested
	fclose(fp);
	return true;
}

// return two letters indicating DISPLAY or INTERNAL,
// and VARIABLE or FIXED. Static buffer, not thread safe
const char* ImageDisk::GetAttributes(int nType) {
	// this is a hacky way to allow up to 3 calls on a single line ;) not thread-safe though!
	static char szBuf[3][3];
	static int cnt=0;

	if (++cnt == 3) cnt=0;

	szBuf[cnt][0]=(nType&TIFILES_INTERNAL) ? 'I' : 'D';
	szBuf[cnt][1]=(nType&TIFILES_VARIABLE) ? 'V' : 'F';
	szBuf[cnt][2]='\0';

	return szBuf[cnt];
}

// Read a file by sectors (file type irrelevant)
// LengthSectors - number of sectors to read
// csName - filename to read
// DataBuffer - address of data in VDP
// RecordNumber - first sector to read
// If 0 sectors spectified, the following are returned by ReadFileSectors, 
// They are the same as in a PAB.
// FileType, RecordsPerSector, BytesInLastSector, RecordLength, NumberRecords
// On return, LengthSectors must contain the actual number of sectors read,
bool ImageDisk::ReadFileSectors(FileInfo *pFile) {
	unsigned char tmpbuf[256];

	if (pFile->LengthSectors == 0) {
		FileInfo lclFile;
		lclFile.CopyFileInfo(pFile, true);
		if (!TryOpenFile(&lclFile)) {
			pFile->LastError = ERR_FILEERROR;
			return false;
		}

		pFile->LengthSectors = lclFile.LengthSectors;
		pFile->FileType = lclFile.FileType;
		pFile->RecordsPerSector = lclFile.RecordsPerSector;
		pFile->BytesInLastSector = lclFile.BytesInLastSector;
		pFile->RecordLength = lclFile.RecordLength;
		pFile->NumberRecords = lclFile.NumberRecords;

		debug_write("Information request on file %s (Type >%02x, %d sectors, Records Per Sector %d, EOF Offset %d, Record Length %d, Number Records %d)", 
			pFile->csName, pFile->FileType, pFile->LengthSectors, pFile->RecordsPerSector, pFile->BytesInLastSector, pFile->RecordLength, pFile->NumberRecords);

		return true;
	}

	// sanity test
	if (pFile->LengthSectors*256 + pFile->DataBuffer > 0x4000) {
		debug_write("Attempt to sector read file %s past end of VDP, truncating.", pFile->csName);
		pFile->LengthSectors = (0x4000 - pFile->DataBuffer) / 256;
		if (pFile->LengthSectors < 1) {
			debug_write("Not enough VDP RAM for even one sector, aborting.");
			pFile->LastError = ERR_BUFFERFULL;
			return false;
		}
	}

	debug_write("Reading drive %d file %s sector %d-%d to VDP %04x", pFile->nDrive, pFile->csName, pFile->RecordNumber, pFile->RecordNumber+pFile->LengthSectors-1, pFile->DataBuffer);

	// we just want to read by sector, so we can reuse some of the buffering code
	CString csFileName=BuildFilename(pFile);
	FILE *fp=fopen(csFileName, "rb");
	if (NULL == fp) {
		debug_write("Failed to open %s", (LPCSTR)csFileName);
		return false;
	}

	// read in the FDR so we can get the cluster list
	if (!FindFileFDR(fp, pFile, tmpbuf)) {
		fclose(fp);
		debug_write("Failed to retrieve FDR at sector %d for %s", pFile->nLocalData, csFileName);
		return false;
	}

	// Get the sector list by parsing the Cluster list (0 terminated)
	// Then we don't need the FDR anymore.
	int *pSectorList = ParseClusterList(tmpbuf);
	if (NULL == pSectorList) {
		fclose(fp);
		debug_write("Failed to parse cluster list for %s", csFileName);
		return false;
	}

	// now we just read the desired sectors. We do need to make sure we don't run off the end
	int nLastSector=0;
	while (pSectorList[nLastSector] != 0) nLastSector++;
	int nVDPAdr = pFile->DataBuffer;

	for (int idx=pFile->RecordNumber; idx < pFile->RecordNumber + pFile->LengthSectors; idx++) {
		if (idx >= nLastSector) {
			debug_write("Reading out of range... aborting.");
			break;
		}
		GetSectorFromDisk(fp, pSectorList[idx], &VDP[nVDPAdr]);
		nVDPAdr+=256;
	}
	free(pSectorList);
	fclose(fp);

	return true;
}
